// ====== UI Interactions ======

import { debounce, throttle } from '../core/utils.js';
import { $, $$, showNotification } from '../core/helpers.js';

class UIManager {
    constructor() {
        this.modals = new Map();
        this.dropdowns = new Map();
        this.init();
    }

    init() {
        this.setupGlobalEventListeners();
        this.setupSmoothScrolling();
        this.setupBackToTop();
    }

    setupGlobalEventListeners() {
        // Close dropdowns when clicking outside
        document.addEventListener('click', (e) => {
            this.dropdowns.forEach((dropdown, trigger) => {
                if (!trigger.contains(e.target) && !dropdown.contains(e.target)) {
                    dropdown.classList.remove('active');
                }
            });
        });

        // Escape key to close modals and dropdowns
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeAllModals();
                this.closeAllDropdowns();
            }
        });
    }

    setupSmoothScrolling() {
        // Smooth scroll for anchor links
        document.addEventListener('click', (e) => {
            const link = e.target.closest('a[href^="#"]');
            if (link) {
                e.preventDefault();
                const target = document.querySelector(link.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({ behavior: 'smooth' });
                }
            }
        });
    }

    setupBackToTop() {
        const backToTop = $('#back-to-top');
        if (!backToTop) return;

        const toggleVisibility = throttle(() => {
            if (window.pageYOffset > 300) {
                backToTop.classList.add('visible');
            } else {
                backToTop.classList.remove('visible');
            }
        }, 100);

        backToTop.addEventListener('click', () => {
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });

        window.addEventListener('scroll', toggleVisibility);
    }

    // Modal management
    registerModal(trigger, modal) {
        this.modals.set(trigger, modal);

        trigger.addEventListener('click', () => {
            this.openModal(modal);
        });

        // Close on overlay click
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                this.closeModal(modal);
            }
        });

        // Close on close button
        const closeBtn = modal.querySelector('.modal-close');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => this.closeModal(modal));
        }
    }

    openModal(modal) {
        modal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }

    closeModal(modal) {
        modal.classList.remove('active');
        document.body.style.overflow = '';
    }

    closeAllModals() {
        this.modals.forEach(modal => this.closeModal(modal));
    }

    // Dropdown management
    registerDropdown(trigger, dropdown) {
        this.dropdowns.set(trigger, dropdown);

        trigger.addEventListener('click', (e) => {
            e.stopPropagation();
            this.toggleDropdown(dropdown);
        });
    }

    toggleDropdown(dropdown) {
        dropdown.classList.toggle('active');
    }

    closeAllDropdowns() {
        this.dropdowns.forEach(dropdown => {
            dropdown.classList.remove('active');
        });
    }

    // Toast/Notification system
    showToast(message, type = 'info', duration = 5000) {
        return showNotification(message, type, duration);
    }

    // Loading states
    showLoading(selector = 'body') {
        const element = $(selector);
        if (element) {
            element.classList.add('loading');
        }
    }

    hideLoading(selector = 'body') {
        const element = $(selector);
        if (element) {
            element.classList.remove('loading');
        }
    }

    // Tab system
    initTabs(container) {
        const tabs = $$('.tab-button', container);
        const contents = $$('.tab-content', container);

        tabs.forEach(tab => {
            tab.addEventListener('click', () => {
                // Deactivate all tabs
                tabs.forEach(t => t.classList.remove('active'));
                contents.forEach(c => c.classList.remove('active'));

                // Activate current tab
                tab.classList.add('active');
                const contentId = tab.getAttribute('data-tab');
                const content = $(`#${contentId}`, container);
                if (content) content.classList.add('active');
            });
        });
    }

    // Accordion system
    initAccordions(container) {
        const accordions = $$('.accordion-header', container);

        accordions.forEach(header => {
            header.addEventListener('click', () => {
                const accordion = header.parentElement;
                const content = header.nextElementSibling;

                accordion.classList.toggle('active');
                
                if (accordion.classList.contains('active')) {
                    content.style.maxHeight = content.scrollHeight + 'px';
                } else {
                    content.style.maxHeight = '0';
                }
            });
        });
    }
}

// Create singleton instance
export const uiManager = new UIManager();