// ====== Header Component Functionality ======

import { $, $$, domReady } from '../core/helpers.js';
import { authManager } from '../core/auth.js';
import { uiManager } from '../modules/ui.js';

class HeaderManager {
    constructor() {
        this.header = null;
        this.mobileMenuBtn = null;
        this.mobileMenu = null;
        this.init();
    }

    async init() {
        await domReady();
        this.setupHeader();
        this.setupMobileMenu();
        this.setupUserMenu();
        this.setupScrollEffects();
    }

    setupHeader() {
        this.header = $('.auth-header');
        if (!this.header) return;

        this.mobileMenuBtn = $('#mobileMenuBtn');
        this.mobileMenu = $('#mobileMenu');
    }

    setupMobileMenu() {
        if (!this.mobileMenuBtn || !this.mobileMenu) return;

        this.mobileMenuBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            this.toggleMobileMenu();
        });

        // Close mobile menu when clicking outside
        document.addEventListener('click', (e) => {
            if (this.mobileMenu.classList.contains('active') &&
                !this.mobileMenu.contains(e.target) &&
                !this.mobileMenuBtn.contains(e.target)) {
                this.closeMobileMenu();
            }
        });

        // Close mobile menu when clicking on links
        $$('.mobile-nav-link', this.mobileMenu).forEach(link => {
            link.addEventListener('click', () => {
                this.closeMobileMenu();
            });
        });
    }

    toggleMobileMenu() {
        this.mobileMenu.classList.toggle('active');
        this.mobileMenuBtn.classList.toggle('active');

        // Update aria-expanded attribute
        const isExpanded = this.mobileMenu.classList.contains('active');
        this.mobileMenuBtn.setAttribute('aria-expanded', isExpanded);

        // Toggle body scroll lock
        if (isExpanded) {
            document.body.style.overflow = 'hidden';
        } else {
            document.body.style.overflow = '';
        }
    }

    closeMobileMenu() {
        this.mobileMenu.classList.remove('active');
        this.mobileMenuBtn.classList.remove('active');
        this.mobileMenuBtn.setAttribute('aria-expanded', 'false');
        document.body.style.overflow = '';
    }

    setupUserMenu() {
        const userMenu = $('.user-menu');
        if (!userMenu) return;

        // Update user greeting
        const greeting = userMenu.querySelector('.user-greeting');
        if (greeting && authManager.user) {
            greeting.textContent = `Welcome, ${authManager.user.name || authManager.user.email}`;
        }

        // Setup logout functionality
        const logoutBtn = userMenu.querySelector('a[href*="logout"]');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', async (e) => {
                e.preventDefault();
                await authManager.logout();
            });
        }
    }

    setupScrollEffects() {
        if (!this.header) return;

        const updateHeader = () => {
            const scrolled = window.scrollY > 50;
            
            if (scrolled) {
                this.header.classList.add('scrolled');
            } else {
                this.header.classList.remove('scrolled');
            }
        };

        // Throttle scroll events for performance
        let ticking = false;
        const update = () => {
            updateHeader();
            ticking = false;
        };

        const requestTick = () => {
            if (!ticking) {
                requestAnimationFrame(update);
                ticking = true;
            }
        };

        window.addEventListener('scroll', requestTick, { passive: true });
        updateHeader(); // Initial check
    }

    // Show/hide header (for special pages)
    show() {
        if (this.header) {
            this.header.style.transform = 'translateY(0)';
        }
    }

    hide() {
        if (this.header) {
            this.header.style.transform = 'translateY(-100%)';
        }
    }

    // Update authentication state in header
    updateAuthState() {
        const authLinks = $('.nav-links');
        if (!authLinks) return;

        // This would be called when authentication state changes
        // to update the header accordingly
    }
}

// Create and initialize header manager
export const headerManager = new HeaderManager();