<?php
namespace App\Core;

class Router {
    private $routes = [];
    
    public function __construct(array $routes) {
        $this->routes = $routes;
    }
    
    public function dispatch($uri, $method) {
        $routeKey = "$method $uri";
        
        error_log("Looking for route: $routeKey");
        
        // Exact match first
        if (isset($this->routes[$routeKey])) {
            error_log("Found exact match: $routeKey");
            return $this->executeRoute($this->routes[$routeKey]);
        }
        
        // Try parameterized routes
        foreach ($this->routes as $pattern => $handler) {
            if ($this->matchRoute($pattern, $routeKey)) {
                error_log("Found pattern match: $pattern -> $routeKey");
                return $this->executeRoute($handler);
            }
        }
        
        // 404 Not Found
        error_log("No route found for: $routeKey");
        http_response_code(404);
        
        // Try to show a helpful error page
        if (file_exists(APP_ROOT . '/resources/views/errors/404.php')) {
            include APP_ROOT . '/resources/views/errors/404.php';
        } else {
            echo "404 - Page not found. Requested: $uri";
        }
        
        return false;
    }
    
    private function matchRoute($pattern, $routeKey) {
        // Convert route pattern to regex
        $regex = preg_replace('/\{[^}]+\}/', '[^/]+', $pattern);
        $regex = str_replace('/', '\/', $regex);
        
        return preg_match("/^$regex$/", $routeKey);
    }
    
    private function executeRoute($handler) {
        if (is_array($handler) && count($handler) === 2) {
            $controllerClass = $handler[0];
            $method = $handler[1];
            
            if (class_exists($controllerClass)) {
                $controller = new $controllerClass();
                if (method_exists($controller, $method)) {
                    return $controller->$method();
                } else {
                    throw new \Exception("Method $method not found in $controllerClass");
                }
            } else {
                throw new \Exception("Controller class $controllerClass not found");
            }
        } elseif (is_callable($handler)) {
            return $handler();
        }
        
        throw new \Exception("Invalid route handler");
    }
}